using System;

namespace gov.va.med.vbecs.DAL.VistALink.OpenLibrary
{
	/// <summary>
	/// This class defines listener message handling modes used in VistALink protocol.
	/// Works as a "string enum". 
	/// Specifies how the request handler should operate once the request is received.
	/// </summary>
	/// <remarks>
	/// Be careful changing underlying string values for this "enum": 
	/// they are mapped directly to VistALink messages XML representation 
	/// (used in XML serialization/deserialization). 
	/// </remarks>
	public class HandlerMode
	{
		/// <summary>
		/// Handler will close the connection immediately after processing the request.
		/// </summary>
		public static readonly HandlerMode SingleCall = new HandlerMode( "single call" );

		/// <summary>
		/// Handler will continue to listen for requests until a close message is 
		/// sent or a request message is sent with a mode of 'single call'
		/// </summary>
		public static readonly HandlerMode Singleton = new HandlerMode( "singleton" );

		private string _handlerMode;

		/// <summary>
		/// The only private constructor. It's called upon 
		/// static initialization of "enum" members.
		/// </summary>
		/// <param name="handlerMode">Status handling mode to assign</param>
		private HandlerMode( string handlerMode )
		{
			_handlerMode = handlerMode;
		}

		/// <summary>
		/// Override for standard ToString() method.
		/// </summary>
		/// <returns>
		///		String representation of listener message handling mode as 
		///		defined in VistALink protocol specification
		/// </returns>
		public override string ToString()
		{
			return _handlerMode;
		}

		/// <summary>
		/// Parses a given string and returns appropriate listener message handling mode object.
		/// Will throw ArgumentOutOfRange exception if input string cannot be parsed.
		/// </summary>
		/// <param name="handlerModeString">
		///		String representation for listener message handling mode 
		///		as defined in VistALink protocol
		///	</param>
		/// <returns>
		///		Appropriate listener message handling mode object.
		///	</returns>
		public static HandlerMode Parse( string handlerModeString )
		{
			if( handlerModeString == null )
				throw( new ArgumentNullException( "handlerModeString" ) );

			switch( handlerModeString )
			{
				case "single call":	return SingleCall;
				case "singleton":	return Singleton;
				default:			
					throw( new StringParseException( SR.Exceptions.StringEnumValueNotFound( "Handler mode", handlerModeString ) ) );
			}
		}
	}
}
